<?php

namespace App\Providers;

use Illuminate\Support\ServiceProvider;
use App\Settings\Services\SettingsService;
use App\Ledger\Services\LedgerService;
use App\Auth\Services\OTPService;

class NeoBankServiceProvider extends ServiceProvider
{
    /**
     * Register services.
     */
    public function register(): void
    {
        // Register Settings Service as Singleton
        $this->app->singleton(SettingsService::class, function ($app) {
            return new SettingsService();
        });

        // Register Ledger Service as Singleton
        $this->app->singleton(LedgerService::class, function ($app) {
            return new LedgerService();
        });

        // Register OTP Service
        $this->app->singleton(OTPService::class, function ($app) {
            return new OTPService($app->make(SettingsService::class));
        });

        // Register aliases for easier access
        $this->app->alias(SettingsService::class, 'settings');
        $this->app->alias(LedgerService::class, 'ledger');
        $this->app->alias(OTPService::class, 'otp');
    }

    /**
     * Bootstrap services.
     */
    public function boot(): void
    {
        // Share public settings with all views
        view()->composer('*', function ($view) {
            if (!$this->app->runningInConsole()) {
                try {
                    $settings = $this->app->make(SettingsService::class);
                    $view->with('appSettings', $settings->getPublic());

                    // Share individual settings for easier access
                    $view->with('appName', $settings->appName());
                    $view->with('appLogo', $settings->appLogo());
                    $view->with('appTagline', $settings->get('app.tagline', 'Digital Banking Reimagined'));
                    $view->with('supportPhone', $settings->supportPhone());
                    $view->with('supportEmail', $settings->supportEmail());
                    $view->with('footerText', $settings->get('app.footer_text', '© ' . date('Y') . ' ' . $settings->appName() . '. All rights reserved.'));
                } catch (\Exception $e) {
                    // Database might not be ready yet
                    $view->with('appSettings', []);
                    $view->with('appName', config('app.name', 'NeonBank'));
                    $view->with('appLogo', null);
                    $view->with('appTagline', 'Digital Banking Reimagined');
                    $view->with('supportPhone', null);
                    $view->with('supportEmail', 'support@neobank.com');
                    $view->with('footerText', '© ' . date('Y') . ' NeonBank. All rights reserved.');
                }
            }
        });
    }
}
